USE AdventureWorks2014;
GO
/*
Example execution statement that returns info on AdventureWorks2014.Production.Product, including extended properties in the output.
 EXEC dbo.Document_Schema @Database_Name = 'AdventureWorks2014', @Schema_Name = 'Person', @Table_Name = 'Person', @Print_Results = 1, @Show_Extended_Properties = 1;
*/
IF EXISTS (SELECT * FROM sys.procedures WHERE procedures.name = 'Document_Schema')
BEGIN
	DROP PROCEDURE dbo.Document_Schema;
END
GO

CREATE PROCEDURE dbo.Document_Schema
	@Database_Name SYSNAME,
	@Schema_Name SYSNAME = NULL,
	@Table_Name SYSNAME = NULL,
	@Print_Results BIT = 1, -- When set to 1, will print results, instead of selecting them.
	@Show_Extended_Properties BIT = 1
AS
BEGIN
	SET NOCOUNT ON;
	DECLARE @Sql_Command NVARCHAR(MAX);
	DECLARE @Schema_List NVARCHAR(MAX) = '';
	-- These tables will hold the collection of objects to document
	DECLARE @Schemas TABLE
		(Schema_Name SYSNAME NOT NULL);
	DECLARE @Tables TABLE
		(Schema_Name SYSNAME, Table_Name SYSNAME NOT NULL, Result_Text NVARCHAR(MAX) NULL);

	DECLARE @Columns TABLE
		(Schema_Name SYSNAME NOT NULL, Table_Name SYSNAME NOT NULL, Column_Name SYSNAME NOT NULL, Type_Name SYSNAME NOT NULL, Ordinal_Position SMALLINT NOT NULL,
		 Column_Length SMALLINT NOT NULL, Column_Precision TINYINT NOT NULL, Column_Scale TINYINT NOT NULL, Column_Collation SYSNAME NULL, Is_Nullable BIT NOT NULL,
		 Is_Identity BIT NOT NULL, Is_Computed BIT NOT NULL, is_sparse BIT NOT NULL, Identity_Seed BIGINT NULL, Identity_Increment BIGINT NULL,
		 Default_Constraint_Name SYSNAME NULL, Default_Constraint_Definition NVARCHAR(MAX) NULL, Computed_Column_Definition NVARCHAR(MAX));
	DECLARE @Exec_Dynamic_SQL_Output TABLE
		(String_Output_1 NVARCHAR(MAX), String_Output_2 NVARCHAR(MAX)); -- This is a dummy variable for use when we need to dump the output of dynamic SQL somewhere, but not into a staging table (yet).

	IF @Database_Name IS NULL
	BEGIN
		RAISERROR('Document_Schema: Please provide a database name for this stored procedure', 16, 1);
		RETURN;			
	END
	-- Collect a list of all schemas in the database specified that contain at least one table.
	IF @Schema_Name IS NULL
	BEGIN
		SELECT @Sql_Command = '
			USE [' + @Database_Name + '];
			SELECT DISTINCT
				schemas.name
			FROM sys.schemas
			INNER JOIN sys.tables
			ON schemas.schema_id = tables.schema_id
			WHERE tables.is_ms_shipped = 0;'
		INSERT INTO @Schemas
			(Schema_Name)
		EXEC sp_executesql @Sql_Command;
	END
	ELSE
	BEGIN
		SELECT @Sql_Command = '
			USE [' + @Database_Name + '];
			SELECT DISTINCT
				schemas.name
			FROM sys.schemas
			INNER JOIN sys.tables
			ON schemas.schema_id = tables.schema_id
			WHERE schemas.name = ''' + @Schema_Name + ''';'
		INSERT INTO @Schemas
			(Schema_Name)
		EXEC sp_executesql @Sql_Command;

		IF NOT EXISTS (SELECT * FROM @Schemas)
		BEGIN
			RAISERROR('Document_Schema: The schema name provided does not exist, or it contains no user tables', 16, 1);
			RETURN;			
		END
	END
	-- Convert schema list into a comma separated list, for easy inclusion in dynamic SQL.
	SELECT
		@Schema_List = @Schema_List + '''' + Schema_Name + ''','
	FROM @Schemas;
	SELECT @Schema_List = LEFT(@Schema_List, LEN(@Schema_List) - 1); -- Remove trailing comma.
	-- Collect a list of all tables in the database & schema specified that contain at least one table.
	SELECT @Sql_Command = '
		USE [' + @Database_Name + '];
		SELECT DISTINCT
			schemas.name AS Schema_Name,
			tables.name AS Table_Name
		FROM sys.tables
		INNER JOIN sys.schemas
		ON schemas.schema_id = tables.schema_id
		WHERE tables.is_ms_shipped = 0';
	IF @Table_Name IS NOT NULL
	BEGIN
		SELECT @Sql_Command = @Sql_Command + '
		AND tables.name = ''' + @Table_Name + '''';
	END
	SELECT @Sql_Command = @Sql_Command + '
		AND schemas.name IN (' + @Schema_List + ')';

	INSERT INTO @Tables
		(Schema_Name, Table_Name)
	EXEC sp_executesql @Sql_Command;
	-- If there are no tables in the result set to document, exit the stored procedure with the approriate message.
	IF NOT EXISTS (SELECT * FROM @Tables)
	BEGIN
		RAISERROR('Document_Schema: No tables found for the criteria provided, or no user tables exist in the database.', 16, 1);
		RETURN;
	END
	-- Collect column info for each table, including metadata such as data type and nullability.  It's faster and easier to get all column data and remove whatever we don't need, rather than
	-- build complex, tiered dynamic SQL to sort this out in one step.
	SELECT @Sql_Command = '
		USE [' + @Database_Name + '];
			SELECT
				schemas.name AS Schema_Name,
				tables.name AS Table_Name,
				columns.name AS Column_Name,
				UPPER(types.name) AS Column_Data_Type,
				ROW_NUMBER() OVER (PARTITION BY schemas.name, tables.name ORDER BY columns.column_id ASC) AS Ordinal_Position,
				columns.max_length AS Column_Length,
				columns.precision AS Column_Precision,
				columns.scale AS Column_Scale,
				columns.collation_name AS Column_Collation,
				columns.is_nullable AS Is_Nullable,
				columns.is_identity AS Is_Identity,
				columns.is_computed AS Is_Computed,
				columns.is_sparse AS Is_Sparse,
				CAST(identity_columns.seed_value AS BIGINT) AS Identity_Seed,
				CAST(identity_columns.increment_value AS BIGINT) AS Identity_Increment,
				default_constraints.name AS Default_Constraint_Name,
				UPPER(default_constraints.definition) AS Default_Constraint_Definition,
				UPPER(computed_columns.definition) AS Computed_Column_Definition
			FROM sys.schemas
			INNER JOIN sys.tables
			ON schemas.schema_id = tables.schema_id
			INNER JOIN sys.columns
			ON tables.object_id = columns.object_id
			INNER JOIN sys.types
			ON columns.user_type_id = types.user_type_id
			LEFT JOIN sys.identity_columns
			ON columns.object_id = identity_columns.object_id
			AND columns.column_id = identity_columns.column_id
			LEFT JOIN sys.default_constraints
			ON schemas.schema_id = default_constraints.schema_id
			AND columns.object_id = default_constraints.parent_object_id
			AND columns.column_id = default_constraints.parent_column_id
			LEFT JOIN sys.computed_columns
			ON columns.object_id = computed_columns.object_id
			AND columns.column_id = computed_columns.column_id';
		IF @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE schemas.name = ''' + @Schema_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			AND tables.name = ''' + @Table_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE tables.name = ''' + @Table_Name + '''';
		END

	INSERT INTO @Columns
		(Schema_Name, Table_Name, Column_Name , Type_Name, Ordinal_Position, Column_Length, Column_Precision,
		 Column_Scale, Column_Collation, Is_Nullable, Is_Identity, Is_Computed, is_sparse, Identity_Seed, Identity_Increment,
		 Default_Constraint_Name, Default_Constraint_Definition, Computed_Column_Definition)
	EXEC sp_executesql @Sql_Command;
	DELETE COLUMN_DATA
	FROM @Columns COLUMN_DATA WHERE NOT EXISTS (SELECT * FROM @Tables TABLE_DATA WHERE TABLE_DATA.Schema_Name = COLUMN_DATA.Schema_Name AND TABLE_DATA.Table_Name = COLUMN_DATA.Table_Name);

	-- Generate list of foreign keys to process
	DECLARE @Foreign_Keys TABLE
		(Foreign_Key_Name SYSNAME NOT NULL, Foreign_Key_Schema_Name SYSNAME NOT NULL, Foreign_Key_Table_Name SYSNAME NOT NULL, Foreign_Key_Creation_Script NVARCHAR(MAX) NOT NULL);
	SELECT @Sql_Command = '
	USE [' + @Database_Name + '];

	SELECT 
		FOREIGN_KEY_DATA.name AS Foreign_Key_Name,
		FOREIGN_KEY_SCHEMA.name AS Foreign_Key_Schema_Name,
		FOREIGN_KEY_TABLE.name AS Foreign_Key_Table_Name,
		''ALTER TABLE ['' + FOREIGN_KEY_SCHEMA.name + ''].['' + FOREIGN_KEY_TABLE.name + ''] WITH '' + CASE WHEN FOREIGN_KEY_DATA.is_not_trusted = 1 THEN ''NOCHECK'' ELSE ''CHECK'' END +
							   '' ADD CONSTRAINT ['' + FOREIGN_KEY_DATA.name + '']
FOREIGN KEY('' + STUFF(( SELECT '', '' + FOREIGN_KEY_COLUMN.name
			FROM sys.foreign_keys
			INNER JOIN sys.foreign_key_columns
			ON foreign_keys.object_id = foreign_key_columns.constraint_object_id
			INNER JOIN sys.tables FOREIGN_KEY_TABLE
			ON foreign_keys.parent_object_id = FOREIGN_KEY_TABLE.object_id
			INNER JOIN sys.schemas FOREIGN_KEY_SCHEMA
			ON FOREIGN_KEY_SCHEMA.schema_id = FOREIGN_KEY_TABLE.schema_id
			INNER JOIN sys.columns as FOREIGN_KEY_COLUMN
			ON foreign_key_columns.parent_object_id = FOREIGN_KEY_COLUMN.object_id 
			AND foreign_key_columns.parent_column_id = FOREIGN_KEY_COLUMN.column_id
			WHERE FOREIGN_KEY_DATA.object_id = foreign_keys.object_id
			AND FOREIGN_KEY_DATA.name = foreign_keys.name
			ORDER BY FOREIGN_KEY_TABLE.name, foreign_key_columns.constraint_column_id
			FOR XML PATH('''')), 1, 2, '''') + '')
REFERENCES ['' + FOREIGN_KEY_SCHEMA.name + ''].['' + REFERENCED_TABLE.name + ''] ('' + 
		STUFF(( SELECT '', '' + REFERENECD_COLUMN.name
					FROM sys.foreign_keys
					INNER JOIN sys.foreign_key_columns
					ON foreign_keys.object_id = foreign_key_columns.constraint_object_id
					INNER JOIN sys.tables REFERENCED_TABLE
					ON foreign_keys.referenced_object_id = REFERENCED_TABLE.object_id
					INNER JOIN sys.schemas REFERENCED_KEY_SCHEMA
					ON REFERENCED_KEY_SCHEMA.schema_id = REFERENCED_TABLE.schema_id
					INNER JOIN sys.columns REFERENECD_COLUMN
					ON foreign_key_columns.referenced_object_id = REFERENECD_COLUMN.object_id
					AND foreign_key_columns.referenced_column_id = REFERENECD_COLUMN.column_id
					WHERE FOREIGN_KEY_DATA.object_id = foreign_keys.object_id
					AND FOREIGN_KEY_DATA.name = foreign_keys.name
					ORDER BY REFERENCED_TABLE.name, foreign_key_columns.constraint_column_id
				FOR XML PATH('''')), 1, 2, '''') + ''));
GO''
		AS Foreign_Key_Creation_Script
	FROM sys.foreign_keys FOREIGN_KEY_DATA
	INNER JOIN sys.tables FOREIGN_KEY_TABLE
	ON FOREIGN_KEY_DATA.parent_object_id = FOREIGN_KEY_TABLE.object_id
	INNER JOIN sys.tables REFERENCED_TABLE
	ON FOREIGN_KEY_DATA.referenced_object_id = REFERENCED_TABLE.object_id
	INNER JOIN sys.schemas FOREIGN_KEY_SCHEMA
	ON FOREIGN_KEY_SCHEMA.schema_id = FOREIGN_KEY_TABLE.schema_id
	INNER JOIN sys.schemas REFERENCED_KEY_SCHEMA
	ON REFERENCED_KEY_SCHEMA.schema_id = REFERENCED_TABLE.schema_id';
	INSERT INTO @Foreign_Keys
		(Foreign_Key_Name, Foreign_Key_Schema_Name, Foreign_Key_Table_Name, Foreign_Key_Creation_Script)
	EXEC sp_executesql @Sql_Command;

	DECLARE @Schema_Name_Current NVARCHAR(MAX);
	DECLARE @Table_Name_Current NVARCHAR(MAX);
	-- Collects info on check constraints
	DECLARE @Check_Constraints TABLE
		(Schema_Name SYSNAME, Table_Name SYSNAME, Check_Constraint_Definition NVARCHAR(MAX));
	SELECT @Sql_Command = '
		USE [' + @Database_Name + '];
		SELECT
			schemas.name AS Schema_Name,
			tables.name AS Table_Name,
			''ALTER TABLE ['' + schemas.name + ''].['' + tables.name + '']
WITH '' + CASE WHEN check_constraints.is_not_trusted = 1 THEN ''NOCHECK'' ELSE ''CHECK'' END + '' ADD CONSTRAINT ['' + check_constraints.name + '']
CHECK '' + check_constraints.definition
		FROM sys.check_constraints
		INNER JOIN sys.tables
		ON tables.object_id = check_constraints.parent_object_id
		INNER JOIN sys.schemas
		ON tables.schema_id = schemas.schema_id';
		IF @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE schemas.name = ''' + @Schema_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			AND tables.name = ''' + @Table_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE tables.name = ''' + @Table_Name + '''';
		END

	INSERT INTO @Check_Constraints
		(Schema_Name, Table_Name, Check_Constraint_Definition)
	EXEC sp_executesql @Sql_Command;
	-- Collect into on indexes & primary keys
	DECLARE @Indexes TABLE
		(Index_Name SYSNAME NOT NULL, Schema_Name SYSNAME NOT NULL, Table_Name SYSNAME NOT NULL, Is_Unique BIT NOT NULL, Has_Filter BIT NOT NULL, Filter_Definition NVARCHAR(MAX) NULL, Index_Type NVARCHAR(MAX) NOT NULL, Index_Column_List NVARCHAR(MAX) NULL, Include_Column_List NVARCHAR(MAX) NULL, Is_Primary_Key BIT NOT NULL);
	SELECT @Sql_Command = '
		USE [' + @Database_Name + '];

		WITH CTE_INDEX_COLUMNS AS (
			SELECT
				INDEX_DATA.name AS Index_Name,
				SCHEMA_DATA.name AS Schema_Name,
				TABLE_DATA.name AS Table_Name,
				INDEX_DATA.is_unique,
				INDEX_DATA.has_filter,
				INDEX_DATA.filter_definition,
				INDEX_DATA.type_desc AS Index_Type,
				STUFF(( SELECT '', '' + columns.name + CASE WHEN INDEX_DATA.type_desc <> ''XML'' THEN CASE WHEN index_columns.is_descending_key = 1 THEN '' DESC'' ELSE '' ASC'' END ELSE '''' END
						FROM sys.tables
						INNER JOIN sys.indexes
						ON tables.object_id = indexes.object_id
						INNER JOIN sys.index_columns
						ON indexes.object_id = index_columns.object_id
						AND indexes.index_id = index_columns.index_id
						INNER JOIN sys.columns
						ON tables.object_id = columns.object_id
						AND index_columns.column_id = columns.column_id
						WHERE INDEX_DATA.object_id = indexes.object_id
						AND INDEX_DATA.index_id = indexes.index_id
						AND index_columns.is_included_column = 0
						ORDER BY index_columns.key_ordinal
					FOR XML PATH('''')), 1, 2, '''') AS Index_Column_List,
					STUFF(( SELECT '', '' + columns.name
						FROM sys.tables
						INNER JOIN sys.indexes
						ON tables.object_id = indexes.object_id
						INNER JOIN sys.index_columns
						ON indexes.object_id = index_columns.object_id
						AND indexes.index_id = index_columns.index_id
						INNER JOIN sys.columns
						ON tables.object_id = columns.object_id
						AND index_columns.column_id = columns.column_id
						WHERE INDEX_DATA.object_id = indexes.object_id
						AND INDEX_DATA.index_id = indexes.index_id
						AND index_columns.is_included_column = 1
						ORDER BY index_columns.key_ordinal
					FOR XML PATH('''')), 1, 2, '''') AS Include_Column_List,
				Is_Primary_Key
			FROM sys.indexes INDEX_DATA
			INNER JOIN sys.tables TABLE_DATA
			ON TABLE_DATA.object_id = INDEX_DATA.object_id
			INNER JOIN sys.schemas SCHEMA_DATA
			ON TABLE_DATA.schema_id = SCHEMA_DATA.schema_id'
		IF @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE SCHEMA_DATA.name = ''' + @Schema_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			AND TABLE_DATA.name = ''' + @Table_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE tables.name = ''' + @Table_Name + '''';
		END
		SELECT @Sql_Command = @Sql_Command + ')
		SELECT
			Index_Name,
			Schema_Name,
			Table_Name,
			is_unique,
			has_filter,
			filter_definition,
			Index_Type,
			Index_Column_List,
			ISNULL(Include_Column_List, '''') AS Include_Column_List,
			Is_Primary_Key
		FROM CTE_INDEX_COLUMNS
		WHERE CTE_INDEX_COLUMNS.Index_Type <> ''HEAP''';
	INSERT INTO @Indexes
		(Index_Name, Schema_Name, Table_Name, Is_Unique, Has_Filter, Filter_Definition, Index_Type, Index_Column_List, Include_Column_List, Is_Primary_Key)
	EXEC sp_executesql @Sql_Command;
	-- Collect info on triggers
	DECLARE @Triggers TABLE
		(Schema_Name SYSNAME NOT NULL, Table_Name SYSNAME NOT NULL, Trigger_Definition NVARCHAR(MAX) NOT NULL);
	SELECT @Sql_Command = '
		USE [' + @Database_Name + '];
		SELECT
			schemas.name AS Schema_Name,
			tables.name AS Table_Name,
			sql_modules.definition AS Trigger_Definition
		FROM sys.triggers
		INNER JOIN sys.sql_modules
		ON triggers.object_id = sql_modules.object_id
		INNER JOIN sys.tables
		ON triggers.parent_id = tables.object_id
		INNER JOIN sys.schemas
		ON schemas.schema_id = tables.schema_id';
		IF @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE schemas.name = ''' + @Schema_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			AND tables.name = ''' + @Table_Name + '''';
		END
		IF @Table_Name IS NOT NULL AND @Schema_Name IS NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			WHERE tables.name = ''' + @Table_Name + '''';
		END

	INSERT INTO @Triggers
		(Schema_Name, Table_Name, Trigger_Definition)
	EXEC sp_executesql @Sql_Command;

	-- Extended Properties
	DECLARE @Extended_Property TABLE
		(Object_Type NVARCHAR(60) NOT NULL, Extended_Property_Name SYSNAME NOT NULL, Extended_Property_Value NVARCHAR(MAX) NOT NULL, Schema_Name SYSNAME NOT NULL, Object_Name SYSNAME NOT NULL, Parent_Object_Name SYSNAME NULL, Parent_Column_Name SYSNAME NULL, Index_Name SYSNAME NULL);
	IF @Show_Extended_Properties = 1
	BEGIN
		SELECT @Sql_Command = '
			USE [' + @Database_Name + '];
			SELECT
				Child.type_desc AS Object_Type,
				extended_properties.name AS Extended_Property_Name,
				CAST(extended_properties.value AS NVARCHAR(MAX)) AS Extended_Property_Value,
				schemas.name AS Schema_Name,
				Child.name AS Object_Name,
				Parent.name AS Parent_Object_Name,
				columns.name AS Parent_Column_Name,
				indexes.name AS Index_Name
			FROM sys.extended_properties
			INNER JOIN sys.objects Child
			ON extended_properties.major_id = Child.object_id
			INNER JOIN sys.schemas
			ON schemas.schema_id = Child.schema_id
			LEFT JOIN sys.objects Parent
			ON Parent.object_id = Child.parent_object_id
			LEFT JOIN sys.columns
			ON Child.object_id = columns.object_id
			AND extended_properties.minor_id = columns.column_id
			AND extended_properties.class_desc = ''OBJECT_OR_COLUMN''
			AND extended_properties.minor_id <> 0
			LEFT JOIN sys.indexes
			ON Child.object_id = indexes.object_id
			AND extended_properties.minor_id = indexes.index_id
			AND extended_properties.class_desc = ''INDEX''
			WHERE Child.type_desc IN (''CHECK_CONSTRAINT'', ''DEFAULT_CONSTRAINT'', ''FOREIGN_KEY_CONSTRAINT'', ''PRIMARY_KEY_CONSTRAINT'', ''SQL_TRIGGER'', ''USER_TABLE'')'			
		IF @Schema_Name IS NOT NULL
		BEGIN
			SELECT @Sql_Command = @Sql_Command + '
			AND schemas.name = ''' + @Schema_Name + '''';
		END

		SELECT @Sql_Command = @Sql_Command + '
ORDER BY Child.type_desc ASC';

		INSERT INTO @Extended_Property
			(Object_Type, Extended_Property_Name, Extended_Property_Value, Schema_Name, Object_Name, Parent_Object_Name, Parent_Column_Name, Index_Name)
		EXEC sp_executesql @Sql_Command;
	END
	-- Output the results
	DECLARE table_cursor CURSOR FOR SELECT Schema_Name, Table_Name FROM @Tables;
	DECLARE @Schema_Build_Text NVARCHAR(MAX);
	OPEN table_cursor;
	FETCH NEXT FROM table_cursor INTO @Schema_Name_Current, @Table_Name_Current;
	WHILE @@FETCH_STATUS = 0
	BEGIN
		SELECT @Schema_Build_Text = 'USE [' + @Database_Name + '];' + '

' +
		'CREATE TABLE [' + TABLE_DATA.Schema_Name + '].[' + TABLE_DATA.Table_Name + ']('
		FROM @Tables TABLE_DATA
		WHERE TABLE_DATA.Schema_Name = @Schema_Name_Current
		AND TABLE_DATA.Table_Name = @Table_Name_Current
				
		SELECT
			@Schema_Build_Text = @Schema_Build_Text + '
	' + COLUMN_DATA.Column_Name + ' ' + 
			CASE WHEN COLUMN_DATA.Is_Computed = 1 THEN '' ELSE -- Don't add metadata if a column is computed (just include definition)
			COLUMN_DATA.Type_Name + -- Basic column metadata
			CASE WHEN COLUMN_DATA.Type_Name = 'DECIMAL' THEN '(' + CAST(COLUMN_DATA.Column_Precision AS NVARCHAR(MAX)) + ',' + CAST(COLUMN_DATA.Column_Scale AS NVARCHAR(MAX)) + ')' ELSE '' END + -- Column precision (decimal)
			CASE WHEN COLUMN_DATA.Type_Name IN ('VARCHAR', 'NVARCHAR', 'NCHAR', 'CHAR') THEN '(' + CAST(COLUMN_DATA.Column_Length AS NVARCHAR(MAX)) + ')' ELSE '' END + -- Column length (string)
			CASE WHEN COLUMN_DATA.is_sparse = 1 THEN ' SPARSE' ELSE '' END + -- If a column is sparse, denote that here.
			CASE WHEN COLUMN_DATA.Is_Identity = 1 THEN ' IDENTITY(' + CAST(Identity_Seed AS NVARCHAR(MAX)) + ',' + CAST(Identity_Increment AS NVARCHAR(MAX)) + ')' ELSE '' END + -- Identity Metadata (optional)
			CASE WHEN COLUMN_DATA.Is_Nullable = 1 THEN ' NULL' ELSE ' NOT NULL' END + -- NULL/NOT NULL definition
			CASE WHEN COLUMN_DATA.Default_Constraint_Name IS NOT NULL THEN ' CONSTRAINT ' + COLUMN_DATA.Default_Constraint_Name + ' DEFAULT ' + COLUMN_DATA.Default_Constraint_Definition ELSE '' END END + -- Default constraint definition (optional)
			CASE WHEN COLUMN_DATA.Is_Computed = 1 THEN 'AS ' + COLUMN_DATA.Computed_Column_Definition ELSE '' END + ','
		FROM @Columns COLUMN_DATA
		WHERE COLUMN_DATA.Table_Name = @Table_Name_Current
		AND COLUMN_DATA.Schema_Name = @Schema_Name_Current
		ORDER BY COLUMN_DATA.Ordinal_Position ASC;

		SELECT @Schema_Build_Text = LEFT(@Schema_Build_Text, LEN(@Schema_Build_Text) - 1);
		-- If the table has a primary key defined, add it here.
		IF EXISTS (SELECT * FROM @Indexes PRIMARY_KEY_DATA WHERE PRIMARY_KEY_DATA.Schema_Name = @Schema_Name_Current AND PRIMARY_KEY_DATA.Table_Name = @Table_Name_Current AND PRIMARY_KEY_DATA.Is_Primary_Key = 1)
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text + ',
CONSTRAINT ' + PRIMARY_KEY_DATA.Index_Name + ' PRIMARY KEY' + -- Primary key name
				Index_Type + -- Clustered vs. Nonclustered key.
				'(' + PRIMARY_KEY_DATA.Index_Column_List + ')' -- Column list.
			FROM @Indexes PRIMARY_KEY_DATA
			WHERE PRIMARY_KEY_DATA.Schema_Name = @Schema_Name_Current
			AND PRIMARY_KEY_DATA.Table_Name = @Table_Name_Current
			AND PRIMARY_KEY_DATA.Is_Primary_Key = 1;
		END

		SELECT @Schema_Build_Text = @Schema_Build_Text + ');
GO';
		-- Populate foreign key data
		IF EXISTS (SELECT * FROM @Foreign_Keys WHERE Foreign_Key_Schema_Name = @Schema_Name_Current AND Foreign_Key_Table_Name = @Table_Name_Current)
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text + '

' + Foreign_Key_Creation_Script
			FROM @Foreign_Keys
			WHERE Foreign_Key_Schema_Name = @Schema_Name_Current
			AND Foreign_Key_Table_Name = @Table_Name_Current;
		END
		-- Populate check constraint data
		IF EXISTS (SELECT * FROM @Check_Constraints WHERE Schema_Name = @Schema_Name_Current AND Table_Name = @Table_Name_Current)
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text + '

' + Check_Constraint_Definition + '
GO'
			FROM @Check_Constraints
			WHERE Schema_Name = @Schema_Name_Current
			AND Table_Name = @Table_Name_Current;
		END
		-- Populate index data
		IF EXISTS (SELECT * FROM @Indexes WHERE Schema_Name = @Schema_Name_Current AND Table_Name = @Table_Name_Current)
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text + '

CREATE ' + CASE WHEN Is_Unique = 1 THEN 'UNIQUE ' ELSE '' END + CASE WHEN Index_Type = 'XML' THEN 'PRIMARY ' ELSE '' END + Index_Type + ' INDEX [' + Index_Name + '] ON [' + Schema_Name + '].[' + Table_Name + ']
(' + Index_Column_List + ')' + CASE WHEN Include_Column_List <> '' THEN '
INCLUDE (' + Include_Column_List + ')' ELSE '' END +
CASE WHEN Has_Filter = 1 THEN '
WHERE ' + Filter_Definition ELSE '' END + '
GO'
			FROM @Indexes
			WHERE Schema_Name = @Schema_Name_Current
			AND Table_Name = @Table_Name_Current
			AND Is_Primary_Key = 0;
		END
		-- Populate trigger data
		IF EXISTS (SELECT * FROM @Triggers WHERE Schema_Name = @Schema_Name_Current AND Table_Name = @Table_Name_Current)
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text + '
' + Trigger_Definition + 'GO'
			FROM @Triggers
			WHERE Schema_Name = @Schema_Name_Current
			AND Table_Name = @Table_Name_Current
		END
		-- Populate extended properties
		IF EXISTS (SELECT * FROM @Extended_Property WHERE COALESCE(Parent_Object_Name, Object_Name, '') = @Table_Name_Current AND Schema_Name = @Schema_Name_Current)
		AND @Show_Extended_Properties = 1
		BEGIN
			SELECT
				@Schema_Build_Text = @Schema_Build_Text +
				'

EXEC sys.sp_addextendedproperty @name = ''' + Extended_Property_Name + ''', @value = ''' + Extended_Property_Value + ''', @level0type = ''SCHEMA'', @level0name = ''' +
				Schema_Name + ''', @level1type = ''TABLE'', @level1name = ''' + COALESCE(Parent_Object_Name, Object_Name, '') + '''' + CASE WHEN NOT(Object_Type = 'USER_TABLE' AND Parent_Object_Name IS NULL AND Parent_Column_Name IS NULL AND Index_Name IS NULL) THEN
				', @level2type = ''' + CASE WHEN Parent_Column_Name IS NOT NULL THEN 'COLUMN'
											WHEN Index_Name IS NOT NULL THEN 'INDEX'
											WHEN Object_Type LIKE '%CONSTRAINT%' THEN 'CONSTRAINT'
											WHEN Object_Type LIKE '%TRIGGER%' THEN 'TRIGGER'
									   END +
				''', @level2name = ''' + COALESCE(Parent_Column_Name, Index_Name, Object_Name) + ''''
				ELSE '' END + ';
GO'
			FROM @Extended_Property
			WHERE COALESCE(Parent_Object_Name, Object_Name, '') = @Table_Name_Current
			AND Schema_Name = @Schema_Name_Current;
		END

		UPDATE TABLE_DATA
			SET Result_Text = @Schema_Build_Text
		FROM @Tables TABLE_DATA
		WHERE TABLE_DATA.Schema_Name = @Schema_Name_Current
		AND TABLE_DATA.Table_Name = @Table_Name_Current;

		FETCH NEXT FROM table_cursor INTO @Schema_Name_Current, @Table_Name_Current;
	END
	CLOSE table_cursor;
	DEALLOCATE table_cursor;
	IF @Print_Results = 0
	BEGIN
		SELECT
			*
		FROM @Tables;
	END
	ELSE
	BEGIN
		DECLARE table_cursor CURSOR FOR SELECT Result_Text FROM @Tables;
		DECLARE @Result_Text NVARCHAR(MAX);
		OPEN table_cursor;
		FETCH NEXT FROM table_cursor INTO @Result_Text;
		WHILE @@FETCH_STATUS = 0
		BEGIN
			PRINT @Result_Text
			FETCH NEXT FROM table_cursor INTO @Result_Text;
		END
		CLOSE table_cursor;
		DEALLOCATE table_cursor;
	END
END
GO
